
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
      *  \brief This file includes the HASH ATP tests
   *
   *  \version CRYS_SEP_ATP_HASH_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern CRYSError_t      TST_ATP_Error;
/************* Private function prototype ***********************/
/************************ Public Functions ******************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   char         *FailureFunctionNameOut_ptr);

/***********************************************************************
* Function Name:                                                           
*  SEPQA_HASH_NotIntegrated         
* 
* Inputs:
* OperationMode - The operation mode : MD5 or SHA1.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes HASH not integrated test
* Algorithm:   
* 1. Allocate memory buffers
* 2. Call the CRYS_HASH_Init function to start the hash process
* 3. Call the CRYS_HASH_Update - This function is called N times
*                                (N = number of blocks in data each block is 16byte)
* 4. Call the CRYS_HASH_Finish to end the HASH process and get the Digest result
* 5. Compare the outputted result to the expected one                                                   
* 6. Free allocated buffers
*
**************************************************************************************/                                
CRYSError_t SEPQA_HASH_NotIntegrated(CRYS_HASH_OperationMode_t   OperationMode,
									 DxUint8_t                  *DataIn_ptr,
									 DxUint32_t                  DataInSize,
									 DxUint8_t                  *ExpectedData_ptr,        
									 DxUint32_t                  DataExpSize,
									 char                       *FailureFunctionNameOut_ptr)
{
	/*** TST Variables Definitions ****/
	DxUint32_t              Block_index ,NumOfBlocks;
	CRYSError_t             TST_TestStatus=0;
	CRYS_HASHUserContext_t  TST_ContextID_ptr;
	DxUint8_t				*pDataIn;
	DxUint8_t				*pDataOut;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(CRYS_HASH_RESULT_SIZE_IN_WORDS*sizeof(DxUint32_t));

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/* 2. calling the HASH init function */
	/*************************************/
	TST_ATP_Error = CRYS_HASH_Init(&TST_ContextID_ptr,
		OperationMode);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_HASH_Init",
		FailureFunctionNameOut_ptr);

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 3. calling the HASH Update function */
	/***************************************/
	NumOfBlocks = DataInSize/TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS;

	for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*HASH block loop*/
	{   

		TST_ATP_Error = CRYS_HASH_Update(&TST_ContextID_ptr,
			pDataIn+(TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS*Block_index),
			TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                       

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
			"CRYS_HASH_Update",
			FailureFunctionNameOut_ptr);	

		if (TST_TestStatus != TESTPASS) 
			goto End;

	}/*End HASH block loop*/


	/* 4. calling the Finish function */
	/**********************************/       
	TST_ATP_Error = CRYS_HASH_Finish(&TST_ContextID_ptr,
									  (DxUint32_t*)pDataOut);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_HASH_Finish",
		FailureFunctionNameOut_ptr);	

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 5. Compare the result to expected result and then Print the expected Vs Outputted*/
	/************************************************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
		ExpectedData_ptr,
		DataExpSize);            


	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut_ptr);								

End:
	/* 6. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;	
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_HASH_Integrated         
* 
* Inputs:
* OperationMode - The operation mode : MD5 or SHA1.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
* This function is qa external application on the SEP, 
* that executes HASH integrated test
* Algorithm:   
* 1. Allocate memory buffers
* 2. Call the CRYS_HASH function to execute the hash process
* 3. Compare the outputted result to the expected one                                                   
* 4. Free allocated buffers
*
**************************************************************************/                                
CRYSError_t SEPQA_HASH_Integrated(CRYS_HASH_OperationMode_t  OperationMode,
								  DxUint8_t                  *DataIn_ptr,
								  DxUint32_t                  DataInSize,
								  DxUint8_t                  *ExpectedData_ptr,        
								  DxUint32_t                  DataExpSize,
								  char                       *FailureFunctionNameOut_ptr)
{
	/*** TST Variables Definitions ****/
	CRYSError_t             TST_TestStatus=0;
	DxUint8_t				*pDataIn;
	DxUint8_t				*pDataOut;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(CRYS_HASH_RESULT_SIZE_IN_WORDS*sizeof(DxUint32_t));

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/* 2. calling the HASH function */
	/********************************/
	TST_ATP_Error = CRYS_HASH(OperationMode,
		pDataIn,
		DataInSize,
		(DxUint32_t*)pDataOut);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_HASH",
		FailureFunctionNameOut_ptr);


	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 3. Compare the result to expected result and then Print the expected Vs Outputted*/
	/************************************************************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut,
		ExpectedData_ptr,
		DataExpSize);            


	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"DX_VOS_MemCmp",
		FailureFunctionNameOut_ptr);								

End:
	/* 4. Free allocated buffers */
	/*****************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;	
}








